<?php
if ( ! defined( 'ABSPATH' ) ) {
    exit;
}

/**
 * Enqueue frontend assets and inline styles
 */
function cleanlite_uc_frontend_assets() {
    if ( is_admin() || is_user_logged_in() ) {
        return;
    }
    
    $enabled = get_option( 'cleanlite_uc_enabled', '1' );
    if ( ! $enabled ) {
        return;
    }

    // 1. Enqueue the base CSS
    wp_enqueue_style( 
        'cleanlite-uc-frontend', 
        CLEANLITE_UC_URL . 'assets/css/uc-frontend.css', 
        array(), 
        CLEANLITE_UC_VERSION 
    );

    // 2. Get Color Options
    $col_head       = get_option( 'cleanlite_uc_color_heading', '#222222' );
    $col_text       = get_option( 'cleanlite_uc_color_text', '#555555' );
    $col_timer      = get_option( 'cleanlite_uc_color_timer', '#333333' );
    $col_overlay    = get_option( 'cleanlite_uc_color_overlay', '#ffffff' );
    $overlay_op     = get_option( 'cleanlite_uc_overlay_opacity', '95' );
    $bg_img         = get_option( 'cleanlite_uc_bg', '' );

    // 3. Convert Hex to RGBA for overlay
    $hex = str_replace( '#', '', $col_overlay );
    if ( strlen( $hex ) === 3 ) {
        $r = hexdec( substr( $hex, 0, 1 ) . substr( $hex, 0, 1 ) );
        $g = hexdec( substr( $hex, 1, 1 ) . substr( $hex, 1, 1 ) );
        $b = hexdec( substr( $hex, 2, 1 ) . substr( $hex, 2, 1 ) );
    } else {
        $r = hexdec( substr( $hex, 0, 2 ) );
        $g = hexdec( substr( $hex, 2, 2 ) );
        $b = hexdec( substr( $hex, 4, 2 ) );
    }
    $rgba_overlay = "rgba({$r}, {$g}, {$b}, " . (int)$overlay_op / 100 . ")";

    // 4. Build Custom CSS (Colors Only - Safe)
    $custom_css = "
        :root {
            --cl-bg-overlay: {$rgba_overlay};
            --cl-col-head: " . sanitize_hex_color($col_head) . ";
            --cl-col-text: " . sanitize_hex_color($col_text) . ";
            --cl-col-timer: " . sanitize_hex_color($col_timer) . ";
        }
        .cl-uc-wrap {
            background-image: url('" . esc_url( $bg_img ) . "');
        }
    ";

    // 5. Inject CSS
    wp_add_inline_style( 'cleanlite-uc-frontend', $custom_css );

    // 6. Enqueue JS
    wp_enqueue_script( 
        'cleanlite-uc-countdown', 
        CLEANLITE_UC_URL . 'assets/js/uc-countdown.js', 
        array(), 
        CLEANLITE_UC_VERSION, 
        true 
    );

    // 7. Pass Data to JS
    $target_ts = 0;
    $date_str = get_option( 'cleanlite_uc_countdown_datetime', '' );
    if ( $date_str ) {
        $target_ts = strtotime( $date_str ); 
    }
    if ( ! $target_ts ) {
        $target_ts = current_time( 'timestamp' ) + WEEK_IN_SECONDS;
    }

    $labels = array(
        'days'    => get_option('cleanlite_uc_label_days', 'Days'),
        'hours'   => get_option('cleanlite_uc_label_hours', 'Hours'),
        'minutes' => get_option('cleanlite_uc_label_minutes', 'Minutes'),
        'seconds' => get_option('cleanlite_uc_label_seconds', 'Seconds'),
    );

    wp_localize_script( 'cleanlite-uc-countdown', 'cleanliteUcData', array(
        'target'  => (int) $target_ts * 1000,
        'enabled' => get_option( 'cleanlite_uc_countdown_enabled', '1' ) ? true : false,
        'labels'  => $labels
    ));
}
add_action( 'wp_enqueue_scripts', 'cleanlite_uc_frontend_assets' );


/**
 * Render maintenance page
 */
function cleanlite_uc_maybe_show_maintenance() {
    if ( is_admin() || is_user_logged_in() ) {
        return;
    }

    if ( ! get_option( 'cleanlite_uc_enabled', '1' ) ) {
        return;
    }

    // Force browsers to see new changes
    nocache_headers();

    $status = get_option( 'cleanlite_uc_status_code', '503' );
    status_header( (int) $status );
    if ( '503' === $status ) {
        header( 'Retry-After: 3600' );
    }

    // Get Content
    $logo           = get_option( 'cleanlite_uc_logo', '' );
    $headline       = get_option( 'cleanlite_uc_headline', 'We are coming soon' );
    $description    = get_option( 'cleanlite_uc_description', 'Our website is under construction.' );
    
    // Style Options (Classes)
    $font_opt       = get_option( 'cleanlite_uc_font_family', 'sans' );
    $timer_style    = get_option( 'cleanlite_uc_timer_style', 'minimal' );
    
    // Body Classes
    $body_classes = 'cl-font-' . esc_attr( $font_opt ); // cl-font-sans, cl-font-serif, etc.
    if( $timer_style === 'boxed' ) {
        $body_classes .= ' cl-timer-boxed';
    }

    ?>
    <!doctype html>
    <html lang="<?php echo esc_attr( get_bloginfo( 'language' ) ); ?>">
    <head>
      <meta charset="utf-8">
      <meta name="viewport" content="width=device-width, initial-scale=1">
      <title><?php echo esc_html( $headline ); ?></title>
      <?php wp_head(); ?>
    </head>
    <body class="<?php echo esc_attr( $body_classes ); ?>">
      <div class="cl-uc-wrap">
        <div class="cl-uc-box">
            <?php if ( $logo ) : ?>
                <div class="cl-uc-logo"><img src="<?php echo esc_url( $logo ); ?>" alt="Logo"></div>
            <?php endif; ?>
            
            <h1 class="cl-uc-headline"><?php echo esc_html( $headline ); ?></h1>
            <div class="cl-uc-desc"><?php echo wp_kses_post( $description ); ?></div>

            <?php if ( get_option( 'cleanlite_uc_countdown_enabled', '1' ) ) : ?>
                <div id="cl-uc-timer"></div>
            <?php endif; ?>

            <?php 
            $fb   = get_option( 'cleanlite_uc_social_facebook' );
            $tw   = get_option( 'cleanlite_uc_social_twitter' );
            $insta= get_option( 'cleanlite_uc_social_instagram' );
            if ( $fb || $tw || $insta ) : ?>
                <div class="cl-uc-socials">
                    <?php if($fb): ?><a href="<?php echo esc_url($fb); ?>" class="cl-uc-social-link">Facebook</a><?php endif; ?>
                    <?php if($tw): ?><a href="<?php echo esc_url($tw); ?>" class="cl-uc-social-link">X / Twitter</a><?php endif; ?>
                    <?php if($insta): ?><a href="<?php echo esc_url($insta); ?>" class="cl-uc-social-link">Instagram</a><?php endif; ?>
                </div>
            <?php endif; ?>

            <div class="cl-uc-login-link">
                <a href="<?php echo esc_url( wp_login_url() ); ?>">Login</a>
            </div>
        </div>
      </div>
      <?php wp_footer(); ?>
    </body>
    </html>
    <?php
    exit;
}
add_action( 'template_redirect', 'cleanlite_uc_maybe_show_maintenance', 1 );